/*
* Creation date : Tues May 29 09:00:00 2007
* Last modified : %modify_time%
*/
/** @file
* \brief This file contains implementation of 
* KDF functions. 
*
* \version CE2_KDF.c#1:csrc:1
* \author Yermalayeu Ihar
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
* All Rights reserved
*/

/************************ Include Files ***********************/

#include "CE2_KDF.h"
#include "LLF_KDF.h"

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Global Data *************************/
/************************ Private function prototype **********/
/************************ Private Functions *******************/
/************************ Public Functions ********************/


/**
****************************************************************
* Function Name: 
*  _DX_KDF_KeyDerivFunc
*
*  @param[in] ZZSecret_ptr 	- A pointer to shared secret value octet string.
*  @param[in] ZZSecretSize  - The shared secret key Size, in bytes.
*  @param[in] OtherInfo     - The pointer to structure, containing pointers 
*                             and sizes of optional data shared by two
*                             entities intended to share the secret value. 
*                             This argument (structure) and also its members 
*                             are optional (if any is not need - set its pointer 
*                             and size to NULL).
*  @param[in] KDFhashMode	  - The hash function to be used. The hash function 
*                             output must be at least 160 bits.
*  @param[out] KeyingData_ptr - A pointer to the keying data derived from the 
*                               secret key, of length KeyLenInBits
*  @param[in] KeyLenInBytes	 - The size in bytes of the keying data to be generated. 
*                              In our implementation - KeyLenInBytes <= 2^32-1  
*                              (in standards KeyLengthInBits < hashlen*(2^32-1)).
*  @param[in] derivation_mode - Specifies one of above described derivation modes.
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - on failure a value from CE2_error.h:
*
* \brief \b 
* Description:
*  _DX_KDF_KeyDerivFunc performs key derivation according to one of some modes 
*  defined in standards: ANSI X9.42-2001, ANSI X9.63, OMA_TS_DRM_DRM_V2_0-20050712-C.
*  The present implementation of the function allows the following operation modes:
*  - CE2_KDF_ASN1_DerivMode - mode based on  ASN.1 DER encoding;
*  - CE2_KDF_ConcatDerivMode - mode based on concatenation;
*  - CE2_KDF_X963_DerivMode = CE2_KDF_ConcatDerivMode;
*  - CE2_KDF_OMADRM_DerivMode - specific mode for OMA DRM.
*  The purpose of this function is to derive a keying data from the shared secret 
*  value and some other optional shared information (SharedInfo).
*  The actual APIs that will be used by the user are:
*  - CE2_KDF_ASN1_KeyDerivFunc ;
*  - CE2_KDF_ConcatKeyDerivFunc ;
*  - CE2_KDF_OMADRM_KeyDerivFunc .
*  
*  \note The length in Bytes of the hash result buffer is denoted by "hashlen".
*  \note All buffers arguments are represented in Big-Endian format.
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_KDF_KeyDerivFunc; 
***************************************************************/
CE2CIMPORT_C CE2Error_t  _DX_KDF_KeyDerivFunc(DxUint8_t                *ZZSecret_ptr,
                                           DxUint32_t                ZZSecretSize,
                                           CE2_KDF_OtherInfo_t     *OtherInfo_ptr,
                                           CE2_KDF_HASH_OpMode_t    KDFhashMode,
                                           CE2_KDF_DerivFuncMode_t  derivation_mode,
                                           DxUint8_t                *KeyingData_ptr,
                                           DxUint32_t                KeyLenInBytes)
{
  if (ZZSecret_ptr == DX_NULL)
    return CE2_KDF_KEY_DERIV_FUNC_ZZSECRET_PTR_ERROR;

  if (OtherInfo_ptr == DX_NULL && derivation_mode == CE2_KDF_ASN1_DerivMode)
    return CE2_KDF_KEY_DERIV_FUNC_OTHERINFO_PTR_ERROR;

  if (KeyingData_ptr == DX_NULL)
    return CE2_KDF_KEY_DERIV_FUNC_KEYINGDATA_PTR_ERROR;

  if (KDFhashMode < 0 || KDFhashMode >= CE2_KDF_HASH_NumOfModes)
    return CE2_KDF_KEY_DERIV_FUNC_KDF_HASH_MODE_ERROR;

  if (derivation_mode < 0 || derivation_mode >= CE2_KDF_DerivFunc_NumOfModes)
    return CE2_KDF_KEY_DERIV_FUNC_DERIVATION_MODE_ERROR;

  return LLF_KDF_KeyDerivFunc(ZZSecret_ptr, ZZSecretSize, OtherInfo_ptr,
     KDFhashMode, derivation_mode, KeyingData_ptr, KeyLenInBytes);
} /* End of _DX_KDF_KeyDerivFunc */
